import pytest
from datetime import datetime, timedelta, date
from sqlalchemy.ext.asyncio import AsyncSession
from database.models import Order
from database.models import Base
from database.queries import async_session
from database.queries import (
    get_reminders,
    get_newly_expired,
    update_order_expiration,
    update_last_reminder,
    mark_as_expired
)



@pytest.mark.asyncio
async def test_get_reminders_and_reset(session: AsyncSession):
    now = datetime.utcnow().date()
    # سفارش با انقضای سه روز بعد
    o = Order(user_id=123, expiration_date=now + timedelta(days=3))
    session.add(o)
    await session.commit()

    reminders = await get_reminders([3])
    assert len(reminders) == 1

    # بعد از update_last_reminder نباید دوباره برگرده
    await update_last_reminder(o.id)
    reminders = await get_reminders([3])
    assert len(reminders) == 0

@pytest.mark.asyncio
async def test_newly_expired_and_mark(session: AsyncSession):
    now = datetime.utcnow().date()
    o = Order(user_id=456, expiration_date=now - timedelta(days=1), status='active')
    session.add(o)
    await session.commit()

    expired = await get_newly_expired()
    assert len(expired) == 1

    await mark_as_expired(o.id)
    expired = await get_newly_expired()
    assert len(expired) == 0

@pytest.mark.asyncio
async def test_update_order_expiration_resets(session: AsyncSession):
    now = datetime.utcnow().date()
    o = Order(user_id=789, expiration_date=now, status='expired', last_reminder_sent=now)
    session.add(o)
    await session.commit()

    new_date = now + timedelta(days=30)
    await update_order_expiration(o.id, new_date, session)
    await session.commit()

    refreshed = (await session.get(Order, o.id))
    assert refreshed.expiration_date == new_date
    assert refreshed.status == 'active'
    assert refreshed.last_reminder_sent is None
