import asyncio
from datetime import datetime, timezone, timedelta
from typing import List

from bot import bot  # از bot.__init__.py ایمپورت می‌کنیم
from aiogram.types import InlineKeyboardMarkup, InlineKeyboardButton

from database.queries import (
    get_cron_enabled,
    get_orders_expiring_within,
    has_notification_been_sent,
    record_notification_sent,
)
from database.models import Order

# notification_type را بر اساس آستانه تعیین کنیم؛ مثال: "EXPIRING_24H"
DEFAULT_NOTIFICATION_TYPE = "EXPIRING_24H"

async def build_renew_keyboard(order_id: int) -> InlineKeyboardMarkup:
    kb = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="♻️ تمدید سرویس", callback_data=f"renew:{order_id}")],
        [InlineKeyboardButton(text="📞 پشتیبانی", callback_data="user:contact_support")]
    ])
    return kb

async def send_expiration_notifications(
    *,
    threshold_seconds: int = 48*3600,
    throttle_seconds: float = 2.0,
    chunk_size: int = 20,
    chunk_pause_seconds: float = 9.0,
    batch_limit: int = 200,
    force_run: bool = False 
) -> dict:
    sent = 0
    failed = 0
    summary = {"sent": 0, "failed": 0}

    cron_enabled = await get_cron_enabled()
    if not cron_enabled:
        return {"skipped": True, "reason": "cron_disabled"}

    from database.db import async_session
    from sqlalchemy import select
    from database.models import Order, PaymentStatus

    offset = 0
    while True:
        orders: List[Order] = await get_orders_expiring_within(threshold_seconds, limit=batch_limit, offset=offset)
        if not orders:
            break

        for order in orders:
            # ⛔️ اگر سفارش تمدید شده باشد (یعنی سفارشی جدید با renew_order_id این سفارش وجود دارد)، از ارسال صرف‌نظر کن
            async with async_session() as session:
                renewed = await session.scalar(
                    select(Order.id).where(
                        Order.renew_order_id == order.id,
                        Order.payment_status == PaymentStatus.approved
                    )
                )
            if renewed:
                continue  # این سفارش تمدید شده، نیاز به اطلاع‌رسانی ندارد

            # بررسی عدم ارسال تکراری
            already = await has_notification_been_sent(order.id, DEFAULT_NOTIFICATION_TYPE)
            if already:
                continue

            try:
                remaining = order.expiration_date - datetime.utcnow()
                remaining_days = remaining.days
                remaining_hours = remaining.seconds // 3600

                text = (
                    f"⏰ اشتراک شما رو به پایان است.\n\n"
                    f"🆔 کد سفارش: {order.order_code}\n"
                    f"⏳ زمان باقیمانده: {remaining_days} روز و {remaining_hours} ساعت\n\n"
                    f"برای تمدید سریع روی دکمه زیر کلیک کنید 👇"
                )
                kb = await build_renew_keyboard(order.id)
                await bot.send_message(order.user_id, text, reply_markup=kb)
                await record_notification_sent(order.id, order.user_id, DEFAULT_NOTIFICATION_TYPE)

                sent += 1
                summary["sent"] = sent

                await asyncio.sleep(throttle_seconds)
                if sent % chunk_size == 0:
                    await asyncio.sleep(chunk_pause_seconds)

            except Exception as e:
                failed += 1
                summary["failed"] = failed
                print(f"Notification send failed for order {order.id} -> {e}")

        if len(orders) < batch_limit:
            break
        offset += batch_limit

    return summary
