from datetime import datetime
from sqlalchemy import select, update
from database.db import async_session
from database.models import UnlimitedRotationGroup, Order
from database.queries import (
    get_available_link,
    increment_link_usage,
    get_link_by_url,
    decrement_link_usage,
)

async def assign_unlimited_group(order_date: datetime) -> UnlimitedRotationGroup | None:
    day = order_date.day - 1
    async with async_session() as session:
        result = await session.execute(
            select(UnlimitedRotationGroup)
            .where(UnlimitedRotationGroup.start_day <= day)
            .order_by(UnlimitedRotationGroup.start_day.desc())
            .limit(1)
        )
        return result.scalar_one_or_none()

async def assign_link_for_order(
    plan_id: int,
    order_date: datetime,
    order_id: int
) -> str | None:
    """
    1) اگر پیش از این لینکی به سفارش اختصاص یافته، آزادش کن.
    2) بعد لینک جدید را پیدا کرده و حجمش را رزرو کن.
    3) URL آن را در سفارش ذخیره کن.
    """
    # --- (۰) بخوان لینک قبلی ---
    async with async_session() as session:
        order = await session.get(Order, order_id)
        old_url = order.assigned_link

    # --- (۱) اگر لینک قبلی بوده، آزادش کن ---
    if old_url:
        old_link = await get_link_by_url(old_url)
        if old_link:
            await decrement_link_usage(old_link.id)

    # --- (۲) تخصیص لینک جدید ---
    grp = await assign_unlimited_group(order_date)
    group_start = grp.start_day if grp else 0

    link = await get_available_link(plan_id, group_start)
    if not link:
        return None

    # افزایش used_users لینک جدید
    await increment_link_usage(link.id)

    # --- (۳) ذخیره URL جدید در سفارش ---
    async with async_session() as session:
        await session.execute(
            update(Order)
            .where(Order.id == order_id)
            .values(assigned_link=link.url)
        )
        await session.commit()

    return link.url
