# bot/utils/cron.py
"""
مدیریت وضعیت کرون اطلاع‌رسانی تمدید اشتراک‌ها
ذخیره در دیتابیس (جدول settings) با پشتیبان فایل cron_status.txt
"""

import os
from sqlalchemy import select, update, insert
from database.db import async_session
from database.models import Setting

STATUS_FILE = "cron_status.txt"
SETTING_KEY = "cron_notify_status"  # کلید وضعیت کرون در جدول settings


async def get_cron_status() -> bool:
    """خواندن وضعیت فعلی کرون (async). اگر در DB نبود از فایل پشتیبان استفاده می‌کنیم."""
    try:
        async with async_session() as session:
            res = await session.execute(select(Setting.value).where(Setting.key == SETTING_KEY))
            row = res.scalar_one_or_none()
            if row is not None:
                return row == "1"

        # اگر در دیتابیس نبود → بررسی فایل پشتیبان (همگام با DB)
        if os.path.exists(STATUS_FILE):
            try:
                with open(STATUS_FILE, "r") as f:
                    return f.read().strip() == "1"
            except Exception:
                return False

    except Exception as e:
        print(f"⚠️ [cron.get_cron_status] error: {e}")

    return False


async def toggle_cron_status() -> bool:
    """تغییر وضعیت کرون (فعال ↔ غیرفعال) و بازگرداندن وضعیت جدید (async)."""
    new_status = not (await get_cron_status())
    value_str = "1" if new_status else "0"

    try:
        async with async_session() as session:
            # وجود رکورد را چک کن
            res = await session.execute(select(Setting).where(Setting.key == SETTING_KEY))
            existing = res.scalar_one_or_none()

            if existing:
                # update
                await session.execute(
                    update(Setting).where(Setting.key == SETTING_KEY).values(value=value_str)
                )
            else:
                # insert
                await session.execute(
                    insert(Setting).values(key=SETTING_KEY, value=value_str)
                )
            await session.commit()

        # به عنوان پشتیبان فایل را هم بنویس
        try:
            with open(STATUS_FILE, "w") as f:
                f.write(value_str)
        except Exception as e:
            print(f"⚠️ [cron.toggle_cron_status] could not write status file: {e}")

        print(f"🔁 وضعیت کرون تغییر کرد → {'🟢 فعال' if new_status else '🔴 غیرفعال'}")
    except Exception as e:
        print(f"⚠️ [cron.toggle_cron_status] DB save error: {e}")

    return new_status
