from aiogram import Router, types
from aiogram.fsm.context import FSMContext
from datetime import datetime

from bot.states.renew import Renew
from bot.keyboards.renew import subscriptions_keyboard, plans_keyboard
from database import queries

router = Router()


@router.callback_query(lambda c: c.data and c.data.startswith("renew:"))
async def start_from_user_cb(cb: types.CallbackQuery, state: FSMContext):
    """
    وقتی کاربر از user.py روی دکمه تمدید کلیک می‌کند.
    این هندلر نقطه ورود تمدید است.
    """
    subs = await queries.get_active_subscriptions(cb.from_user.id)

    if not subs:
        await cb.message.answer("❌ شما هیچ اشتراک فعالی ندارید.")
        return

    if len(subs) == 1:
        # فقط یک اشتراک فعال → مستقیم برویم برای بررسی روزهای باقی‌مانده
        order = subs[0]

        now_date = datetime.utcnow().date()
        exp_date = (
            order.expiration_date.date()
            if hasattr(order.expiration_date, "date")
            else order.expiration_date
        )
        remaining_days = (exp_date - now_date).days

        print("🔥 DEBUG (single sub):", exp_date, now_date, remaining_days)

        if remaining_days > 3:
            await cb.message.answer("⏳ هنوز بیش از ۳ روز اعتبار دارید. امکان تمدید وجود ندارد.")
            return

        await state.update_data(order_id=order.id)
        plans = await queries.get_plans_by_category(order.category_id)

        await cb.message.answer(
            "✅ پلن مورد نظر خود را انتخاب کنید:",
            reply_markup=plans_keyboard(plans),
        )
        await state.set_state(Renew.choosing_plan)
    else:
        # چند اشتراک → کاربر باید انتخاب کند
        await cb.message.answer(
            "📋 لطفاً یکی از اشتراک‌های فعال خود را برای تمدید انتخاب کنید:",
            reply_markup=subscriptions_keyboard(subs),
        )
        await state.set_state(Renew.choosing_subscription)


@router.callback_query(lambda c: c.data and c.data.startswith("renew_order:"))
async def process_renew_callback(cb: types.CallbackQuery, state: FSMContext):
    """
    وقتی کاربر یکی از اشتراک‌ها را انتخاب می‌کند.
    """
    order_id = int(cb.data.split(":")[1])
    order = await queries.get_order_by_id(order_id)

    now_date = datetime.utcnow().date()
    exp_date = (
        order.expiration_date.date()
        if hasattr(order.expiration_date, "date")
        else order.expiration_date
    )
    remaining_days = (exp_date - now_date).days

    print("🔥 DEBUG (multi sub):", exp_date, now_date, remaining_days)

    if remaining_days > 3:
        await cb.message.answer("⏳ هنوز بیش از ۳ روز اعتبار دارید. امکان تمدید وجود ندارد.")
        return

    await state.update_data(order_id=order_id)
    plans = await queries.get_plans_by_category(order.category_id)

    await cb.message.answer(
        "✅ پلن مورد نظر خود را انتخاب کنید:",
        reply_markup=plans_keyboard(plans),
    )
    await state.set_state(Renew.choosing_plan)


@router.callback_query(lambda c: c.data and c.data.startswith("renew_plan:"))
async def process_plan_selection(cb: types.CallbackQuery, state: FSMContext):
    """
    بعد از انتخاب پلن جدید → هدایت به خرید
    """
    plan_id = int(cb.data.split(":")[1])
    data = await state.get_data()
    order_id = data.get("order_id")

    # TODO: اینجا باید کاربر به مسیر خرید هدایت شود
    await cb.message.answer(
        f"🛒 پلن انتخابی شما (ID: {plan_id}) برای تمدید سفارش {order_id} ثبت شد.\n"
        "👉 حالا می‌توانید ادامه خرید را انجام دهید."
    )
    await state.clear()


