# bot/handlers/admin_discount.py
import logging
from typing import Optional
from datetime import datetime

from aiogram import F, Router
from aiogram.types import CallbackQuery, Message, InlineKeyboardMarkup, InlineKeyboardButton
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import StatesGroup, State
from sqlalchemy import select
from sqlalchemy.exc import IntegrityError

from config import ADMINS
from bot.keyboards.admin import discount_menu_keyboard
from database.queries import (
    list_discount_codes,
    get_discount_code,
    delete_discount_code,
    list_discount_usages,
    get_discount_total_sales,
    update_discount_code,   # 👈 اضافه شده
    count_discount_users,
    list_discount_usages_paginated,
)
from database.models import DiscountCode
from database.db import async_session
from bot.utils.helpers import shamsi_to_gregorian, format_shamsi_date

router = Router()
logger = logging.getLogger(__name__)


def is_admin(user_id: int) -> bool:
    return user_id in ADMINS


# ─────────── FSM ───────────
class DiscountFSM(StatesGroup):
    waiting_code = State()
    waiting_percent = State()
    waiting_usage_limit = State()
    waiting_per_user_limit = State()
    waiting_expiration_date = State()


class DiscountSearchFSM(StatesGroup):
    waiting_code = State()


class DiscountEditFSM(StatesGroup):
    waiting_percent = State()
    waiting_usage_limit = State()
    waiting_per_user_limit = State()
    waiting_expiration_date = State()
    target_discount_id = State()


# ─────────── منو ───────────
@router.callback_query(F.data == "admin:discount_menu")
async def open_discount_menu(cb: CallbackQuery):
    await cb.message.edit_text("🎟 منوی مدیریت کدهای تخفیف:", reply_markup=discount_menu_keyboard())
    await cb.answer()


# ─────────── افزودن کد ───────────
@router.callback_query(F.data == "admin:discount:add")
async def add_discount_start(cb: CallbackQuery, state: FSMContext):
    await state.set_state(DiscountFSM.waiting_code)
    await cb.message.edit_text("📝 لطفاً کد تخفیف جدید را وارد کنید (مثلاً: VIP2024):")
    await cb.answer()


@router.message(DiscountFSM.waiting_code)
async def add_discount_code(msg: Message, state: FSMContext):
    await state.update_data(code=msg.text.strip().upper())
    await state.set_state(DiscountFSM.waiting_percent)
    await msg.answer("📊 درصد تخفیف را وارد کنید (مثلاً: 20 برای 20٪):")


@router.message(DiscountFSM.waiting_percent)
async def add_discount_percent(msg: Message, state: FSMContext):
    try:
        percent = int(msg.text.strip())
        if not (1 <= percent <= 100):
            raise ValueError
    except ValueError:
        return await msg.answer("❌ لطفاً یک عدد صحیح بین 1 تا 100 وارد کنید.")

    await state.update_data(percent=percent)
    await state.set_state(DiscountFSM.waiting_usage_limit)
    await msg.answer("🔢 سقف کلی استفاده از کد را وارد کنید (0 = نامحدود):")


@router.message(DiscountFSM.waiting_usage_limit)
async def add_discount_usage_limit(msg: Message, state: FSMContext):
    try:
        limit = int(msg.text.strip())
        usage_limit = None if limit == 0 else limit
    except ValueError:
        return await msg.answer("❌ لطفاً یک عدد صحیح معتبر وارد کنید.")

    await state.update_data(usage_limit=usage_limit)
    await state.set_state(DiscountFSM.waiting_per_user_limit)
    await msg.answer("👤 سقف مصرف هر کاربر را وارد کنید (0 = نامحدود):")


@router.message(DiscountFSM.waiting_per_user_limit)
async def add_discount_per_user_limit(msg: Message, state: FSMContext):
    try:
        limit = int(msg.text.strip())
        per_user_limit = None if limit == 0 else limit
    except ValueError:
        return await msg.answer("❌ لطفاً یک عدد صحیح معتبر وارد کنید.")

    await state.update_data(per_user_limit=per_user_limit)
    await state.set_state(DiscountFSM.waiting_expiration_date)
    await msg.answer("⏳ تاریخ انقضا را وارد کنید (به صورت شمسی YYYY/MM/DD یا 0 برای بدون انقضا):")


@router.message(DiscountFSM.waiting_expiration_date)
async def add_discount_expiration_date(msg: Message, state: FSMContext):
    expires_at = None
    if msg.text.strip() != "0":
        try:
            expires_at = shamsi_to_gregorian(msg.text.strip())
        except Exception:
            return await msg.answer("❌ تاریخ نامعتبر است. لطفاً به فرمت YYYY/MM/DD وارد کنید یا 0 برای بدون انقضا.")

    data = await state.get_data()
    code = data.get("code")
    percent = data.get("percent")
    usage_limit = data.get("usage_limit")
    per_user_limit = data.get("per_user_limit")

    if not code or not percent:
        await state.clear()
        return await msg.answer("⚠️ خطا: داده‌ها ناقص هستند. دوباره اقدام کنید.")

    async with async_session() as session:
        discount = DiscountCode(
            code=code,
            percent=percent,
            usage_limit=usage_limit,
            per_user_limit=per_user_limit,
            expires_at=expires_at,
        )
        session.add(discount)
        try:
            await session.commit()
        except IntegrityError:
            await session.rollback()
            await state.clear()
            return await msg.answer("❌ این کد قبلاً ثبت شده است.")

    txt = [
        "✅ کد تخفیف جدید ذخیره شد:",
        f"🔑 {code}",
        f"📉 {percent}٪",
        f"📦 سقف کلی: {usage_limit if usage_limit else 'نامحدود'}",
        f"👤 سقف هر کاربر: {per_user_limit if per_user_limit else 'نامحدود'}",
        f"⏳ انقضا: {format_shamsi_date(expires_at) if expires_at else 'بدون انقضا'}",
    ]
    await msg.answer("\n".join(txt), reply_markup=discount_menu_keyboard())
    await state.clear()


# ─────────── جستجو ───────────
@router.callback_query(F.data == "admin:discount:search")
async def discount_search_start(cb: CallbackQuery, state: FSMContext):
    await state.set_state(DiscountSearchFSM.waiting_code)
    await cb.message.edit_text("🔍 لطفاً کد تخفیف مورد نظر را وارد کنید:")
    await cb.answer()


@router.message(DiscountSearchFSM.waiting_code)
async def discount_search_process(msg: Message, state: FSMContext):
    code = msg.text.strip().upper()
    disc = await get_discount_code(code)

    if not disc:
        await msg.answer("❌ کدی با این مقدار پیدا نشد.", reply_markup=discount_menu_keyboard())
        await state.clear()
        return

    await _send_discount_details(msg, disc, back_cb="admin:discount_menu")
    await state.clear()


# ─────────── helper: تاریخ ───────────
def _fmt_dt(dt: Optional[datetime]) -> str:
    if not dt:
        return "—"
    return format_shamsi_date(dt)


# ─────────── نمایش جزئیات ───────────
async def _send_discount_details(target, disc: DiscountCode, back_cb: str):
    used_count = getattr(disc, "used_count", 0)
    usage_limit = getattr(disc, "usage_limit", None)
    per_user_limit = getattr(disc, "per_user_limit", None)
    expires_at = getattr(disc, "expires_at", None)

    total_sales = await get_discount_total_sales(disc.id)

    text_lines = [
        f"🎟 <b>جزئیات کد تخفیف</b>",
        f"• کد: <code>{disc.code}</code>",
        f"• درصد: {int(disc.percent)}٪",
        f"• استفاده شده: {used_count}" + (f" / {usage_limit}" if usage_limit else ""),
        f"• سقف هر کاربر: {per_user_limit if per_user_limit else 'نامحدود'}",
        f"• انقضا: {_fmt_dt(expires_at)}",
        f"• 💰 مجموع فروش با این کد: {total_sales:,} تومان",
    ]

    kb = InlineKeyboardMarkup(inline_keyboard=[
        [
            InlineKeyboardButton(text="✏️ ویرایش درصد", callback_data=f"admin:discount:edit:percent:{disc.id}"),
            InlineKeyboardButton(text="✏️ ویرایش سقف کلی", callback_data=f"admin:discount:edit:usage:{disc.id}"),
        ],
        [
            InlineKeyboardButton(text="✏️ ویرایش سقف هر کاربر", callback_data=f"admin:discount:edit:peruser:{disc.id}"),
            InlineKeyboardButton(text="✏️ ویرایش تاریخ انقضا", callback_data=f"admin:discount:edit:expiry:{disc.id}"),
        ],
        [
            InlineKeyboardButton(text="👥 کاربران استفاده‌کننده", callback_data=f"admin:discount:users:{disc.id}:0")
        ],
        [
            InlineKeyboardButton(text="🗑 حذف کد", callback_data=f"admin:discount:delete:{disc.id}"),
            InlineKeyboardButton(text="🔙 بازگشت", callback_data=back_cb),
        ]
    ])

    if isinstance(target, CallbackQuery):
        await target.message.edit_text("\n".join(text_lines), parse_mode="HTML", reply_markup=kb)
    else:
        await target.answer("\n".join(text_lines), parse_mode="HTML", reply_markup=kb)


# ─────────── ویرایش‌ها ───────────
@router.callback_query(F.data.startswith("admin:discount:edit:percent:"))
async def edit_discount_percent_start(cb: CallbackQuery, state: FSMContext):
    discount_id = int(cb.data.split(":")[-1])
    await state.set_state(DiscountEditFSM.waiting_percent)
    await state.update_data(discount_id=discount_id)
    await cb.message.answer("📊 درصد جدید را وارد کنید (1 تا 100):")
    await cb.answer()


@router.message(DiscountEditFSM.waiting_percent)
async def edit_discount_percent(msg: Message, state: FSMContext):
    try:
        percent = int(msg.text.strip())
        if not (1 <= percent <= 100):
            raise ValueError
    except ValueError:
        return await msg.answer("❌ مقدار نامعتبر. یک عدد بین 1 تا 100 بده.")

    data = await state.get_data()
    success = await update_discount_code(data["discount_id"], percent=percent)
    await state.clear()
    await msg.answer("✅ درصد با موفقیت ویرایش شد." if success else "❌ خطا در ویرایش درصد.")


@router.callback_query(F.data.startswith("admin:discount:edit:usage:"))
async def edit_discount_usage_start(cb: CallbackQuery, state: FSMContext):
    discount_id = int(cb.data.split(":")[-1])
    await state.set_state(DiscountEditFSM.waiting_usage_limit)
    await state.update_data(discount_id=discount_id)
    await cb.message.answer("🔢 سقف کلی جدید را وارد کنید (0 = نامحدود):")
    await cb.answer()


@router.message(DiscountEditFSM.waiting_usage_limit)
async def edit_discount_usage(msg: Message, state: FSMContext):
    try:
        val = int(msg.text.strip())
        usage_limit = None if val == 0 else val
    except ValueError:
        return await msg.answer("❌ مقدار نامعتبر.")

    data = await state.get_data()
    success = await update_discount_code(data["discount_id"], usage_limit=usage_limit)
    await state.clear()
    await msg.answer("✅ سقف کلی ویرایش شد." if success else "❌ خطا در ویرایش سقف کلی.")


@router.callback_query(F.data.startswith("admin:discount:edit:peruser:"))
async def edit_discount_peruser_start(cb: CallbackQuery, state: FSMContext):
    discount_id = int(cb.data.split(":")[-1])
    await state.set_state(DiscountEditFSM.waiting_per_user_limit)
    await state.update_data(discount_id=discount_id)
    await cb.message.answer("👤 سقف مصرف هر کاربر را وارد کنید (0 = نامحدود):")
    await cb.answer()


@router.message(DiscountEditFSM.waiting_per_user_limit)
async def edit_discount_peruser(msg: Message, state: FSMContext):
    try:
        val = int(msg.text.strip())
        per_user_limit = None if val == 0 else val
    except ValueError:
        return await msg.answer("❌ مقدار نامعتبر.")

    data = await state.get_data()
    success = await update_discount_code(data["discount_id"], per_user_limit=per_user_limit)
    await state.clear()
    await msg.answer("✅ سقف هر کاربر ویرایش شد." if success else "❌ خطا در ویرایش سقف هر کاربر.")


@router.callback_query(F.data.startswith("admin:discount:edit:expiry:"))
async def edit_discount_expiry_start(cb: CallbackQuery, state: FSMContext):
    discount_id = int(cb.data.split(":")[-1])
    await state.set_state(DiscountEditFSM.waiting_expiration_date)
    await state.update_data(discount_id=discount_id)
    await cb.message.answer("⏳ تاریخ انقضا جدید را وارد کنید (به شمسی YYYY/MM/DD یا 0 برای بدون انقضا):")
    await cb.answer()


@router.message(DiscountEditFSM.waiting_expiration_date)
async def edit_discount_expiry(msg: Message, state: FSMContext):
    text = msg.text.strip()
    expires_at = None
    if text != "0":
        try:
            expires_at = shamsi_to_gregorian(text)
        except Exception:
            return await msg.answer("❌ تاریخ نامعتبر. دوباره تلاش کنید.")

    data = await state.get_data()
    success = await update_discount_code(data["discount_id"], expires_at=expires_at)
    await state.clear()
    await msg.answer(
        f"✅ تاریخ انقضا ویرایش شد: {format_shamsi_date(expires_at) if expires_at else 'بدون انقضا'}"
        if success else "❌ خطا در ویرایش تاریخ انقضا."
    )


# ─────────── لیست، حذف و کاربران بدون تغییر ───────────
@router.callback_query(F.data.startswith("admin:discount:view:"))
async def admin_discount_view(cb: CallbackQuery):
    if not is_admin(cb.from_user.id):
        return await cb.answer("⛔️ شما دسترسی ادمین نداری.", show_alert=True)

    try:
        discount_id = int(cb.data.split(":")[-1])
    except Exception:
        return await cb.answer("❌ شناسه نامعتبر است.", show_alert=True)

    disc = await get_discount_code_by_id(discount_id)
    if not disc:
        return await cb.answer("⚠️ کد یافت نشد.", show_alert=True)

    await _send_discount_details(cb, disc, back_cb="admin:discount:list")
    await cb.answer()


@router.callback_query(F.data.startswith("admin:discount:users:"))
async def admin_discount_users(cb: CallbackQuery):
    if not is_admin(cb.from_user.id):
        return await cb.answer("⛔️ شما دسترسی نداری.", show_alert=True)

    # format: admin:discount:users:{discount_id}:{page}
    parts = cb.data.split(":")
    discount_id = int(parts[3])
    page = int(parts[4]) if len(parts) > 4 else 0

    PAGE_SIZE = 10
    offset = page * PAGE_SIZE

    total_users = await count_discount_users(discount_id)
    total_pages = max((total_users + PAGE_SIZE - 1) // PAGE_SIZE, 1)

    usages = await list_discount_usages_paginated(
        discount_id=discount_id,
        limit=PAGE_SIZE,
        offset=offset
    )

    text = (
        "👥 <b>کاربران استفاده‌کننده از کد تخفیف</b>\n"
        f"📊 مجموع کاربران: <b>{total_users}</b>\n"
        f"📄 صفحه <b>{page + 1}</b> از <b>{total_pages}</b>\n\n"
    )

    if not usages:
        text += "📭 کاربری یافت نشد."
    else:
        for i, u in enumerate(usages, start=offset + 1):
            text += f"{i}. <code>{u.user_id}</code> → {u.count} بار\n"

    buttons = []

    nav_row = []
    if page > 0:
        nav_row.append(
            InlineKeyboardButton(
                text="⬅️ قبلی",
                callback_data=f"admin:discount:users:{discount_id}:{page - 1}"
            )
        )
    if page + 1 < total_pages:
        nav_row.append(
            InlineKeyboardButton(
                text="➡️ بعدی",
                callback_data=f"admin:discount:users:{discount_id}:{page + 1}"
            )
        )

    if nav_row:
        buttons.append(nav_row)

    buttons.append([
        InlineKeyboardButton(
            text="🔙 بازگشت",
            callback_data=f"admin:discount:view:{discount_id}"
        )
    ])

    await cb.message.edit_text(
        text,
        parse_mode="HTML",
        reply_markup=InlineKeyboardMarkup(inline_keyboard=buttons)
    )
    await cb.answer()



def build_discount_list_kb(codes: list) -> InlineKeyboardMarkup:
    buttons = []
    for d in codes:
        view_cb = f"admin:discount:view:{d.id}"
        del_cb = f"admin:discount:delete:{d.id}"
        buttons.append([
            InlineKeyboardButton(text=f"{d.code}  ({d.percent}%)", callback_data=view_cb),
            InlineKeyboardButton(text="🗑 حذف", callback_data=del_cb),
        ])
    buttons.append([InlineKeyboardButton(text="🔙 بازگشت", callback_data="admin:main_menu")])
    return InlineKeyboardMarkup(inline_keyboard=buttons)


@router.callback_query(F.data == "admin:discount:list")
async def admin_discount_list(cb: CallbackQuery):
    if not is_admin(cb.from_user.id):
        return await cb.answer("⛔️ شما دسترسی نداری.", show_alert=True)

    codes = await list_discount_codes()
    if not codes:
        kb = InlineKeyboardMarkup(inline_keyboard=[
            [InlineKeyboardButton(text="🔙 بازگشت", callback_data="admin:main_menu")]
        ])
        await cb.message.edit_text("📭 هیچ کد تخفیفی ثبت نشده.", reply_markup=kb)
        return await cb.answer()

    kb = build_discount_list_kb(codes)
    await cb.message.edit_text("🎟 لیست کدهای تخفیف:", reply_markup=kb)
    await cb.answer()


async def get_discount_code_by_id(discount_id: int) -> Optional[DiscountCode]:
    async with async_session() as session:
        res = await session.execute(select(DiscountCode).where(DiscountCode.id == discount_id))
        return res.scalar_one_or_none()


@router.callback_query(F.data.startswith("admin:discount:delete:"))
async def admin_discount_delete_request(cb: CallbackQuery):
    if not is_admin(cb.from_user.id):
        return await cb.answer("⛔️ شما دسترسی نداری.", show_alert=True)

    discount_id = int(cb.data.split(":")[-1])
    disc = await get_discount_code_by_id(discount_id)
    if not disc:
        return await cb.answer("⚠️ این کد وجود ندارد.", show_alert=True)

    kb = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="✅ بله، حذف کن", callback_data=f"admin:discount:confirm_delete:{discount_id}")],
        [InlineKeyboardButton(text="❌ انصراف", callback_data="admin:discount:list")],
    ])
    await cb.message.edit_text(
        f"⚠️ آیا مطمئنید کد <code>{disc.code}</code> حذف شود؟",
        parse_mode="HTML",
        reply_markup=kb
    )
    await cb.answer()


@router.callback_query(F.data.startswith("admin:discount:confirm_delete:"))
async def admin_discount_confirm_delete(cb: CallbackQuery):
    if not is_admin(cb.from_user.id):
        return await cb.answer("⛔️ شما دسترسی نداری.", show_alert=True)

    discount_id = int(cb.data.split(":")[-1])
    disc = await get_discount_code_by_id(discount_id)
    if not disc:
        return await cb.answer("⚠️ این کد از قبل حذف شده است.", show_alert=True)

    await delete_discount_code(discount_id)
    await cb.message.edit_text(
        f"✅ کد <code>{disc.code}</code> حذف شد.",
        parse_mode="HTML",
        reply_markup=InlineKeyboardMarkup(
            inline_keyboard=[[InlineKeyboardButton(text="🔙 بازگشت", callback_data="admin:discount:list")]]
        )
    )
    await cb.answer("کد حذف شد.")
